<?php
/*
Plugin Name: Site Change Monitor
Description: Sends email alerts for logins, plugin/theme/user changes, and content updates.
Version: 2.0 // Updated version number for batching/async implementation
*/

define('SCM_ALERT_RECIPIENTS', [
    'infrastructure-ops@enlivendesigners.com',
]);

// Define the transient key and batching interval
define('SCM_BATCH_TRANSIENT_KEY', 'scm_queued_alerts');
define('SCM_BATCH_INTERVAL_SECONDS', 300); // 5 minutes (5 * 60 seconds)

// New function to queue alerts for batching
function scm_queue_alert($message_body) {
    // Check if essential functions are available.
    // get_bloginfo is usually fine here, but wp_mail will be used by the cron job.
    if (!function_exists('get_bloginfo')) {
        scm_log_to_file("Attempted to queue alert but get_bloginfo not loaded: " . $message_body);
        return;
    }

    $site_url = get_bloginfo('url');
    $site_name = get_bloginfo('name');

    // Prepend the caution message to the individual alert body.
    // This part goes into the batched email for EACH individual alert.
    $caution_message_part = "Caution: Think carefully before clicking on links or attachments. Never provide User IDs or Passwords. Report any suspicious emails using the 'Report Phishing' button.\n\n";

    // Append the site URL to the individual alert body.
    $message_with_url_part = $message_body . "\n\nAffected Site URL: " . $site_url;

    // The full message for this single event to be queued
    $full_queued_message = $caution_message_part . $message_with_url_part;

    // Get existing queued alerts or initialize an empty array
    $queued_alerts = get_transient(SCM_BATCH_TRANSIENT_KEY);
    if (!is_array($queued_alerts)) {
        $queued_alerts = [];
    }

    // Add the new message with a timestamp and site info for the batch email subject
    $queued_alerts[] = [
        'timestamp' => current_time('mysql'),
        'message'   => $full_queued_message,
        'site_name' => $site_name,
        'site_url'  => $site_url,
    ];

    // Store the updated array back in the transient.
    // The transient will expire after the batch interval, ensuring the cron job picks it up.
    // Added 60 seconds buffer to ensure it's still available when cron runs.
    set_transient(SCM_BATCH_TRANSIENT_KEY, $queued_alerts, SCM_BATCH_INTERVAL_SECONDS + 60);

    // Always log immediately for local debugging/record keeping
    scm_log_to_file($full_queued_message);
}


// Your existing local file logging function
function scm_log_to_file($message) {
    $logfile = WP_CONTENT_DIR . '/scm-change-log.log';
    $timestamp = date('[Y-m-d H:i:s] ');
    error_log($timestamp . $message . "\n", 3, $logfile);
}

// Helper to get user in a safe way
function scm_get_current_username_safe() {
    if (function_exists('wp_get_current_user')) {
        $user = wp_get_current_user();
        return $user->exists() ? $user->user_login : 'Unknown/System';
    }
    return 'Unknown/System (WP Funcs Not Ready)';
}


// --- WP-CRON SCHEDULING ---
// Schedule the custom cron job
add_action('init', 'scm_schedule_batch_sender');
function scm_schedule_batch_sender() {
    // Only schedule if the hook is not already scheduled
    if (!wp_next_scheduled('scm_send_batched_alerts_cron_hook')) {
        wp_schedule_event(time(), 'scm_every_five_minutes', 'scm_send_batched_alerts_cron_hook');
    }
}

// Add a custom cron interval (e.g., every 5 minutes)
add_filter('cron_schedules', 'scm_add_custom_cron_interval');
function scm_add_custom_cron_interval($schedules) {
    $schedules['scm_every_five_minutes'] = array(
        'interval' => SCM_BATCH_INTERVAL_SECONDS,
        'display'  => esc_html__('Every 5 Minutes (SCM)', 'site-change-monitor'),
    );
    return $schedules;
}

// Cron callback to process and send batched alerts
add_action('scm_send_batched_alerts_cron_hook', 'scm_process_queued_alerts');
function scm_process_queued_alerts() {
    // Make sure wp_mail and other essential functions are available for cron run
    if (!function_exists('wp_mail') || !function_exists('get_transient') || !function_exists('delete_transient')) {
        // Log this scenario if the cron job somehow fires too early
        error_log("SCM Error: Core WP mail/transient functions not available during cron run.");
        return;
    }

    // Get all queued alerts
    $queued_alerts = get_transient(SCM_BATCH_TRANSIENT_KEY);

    // If no alerts, nothing to do
    if (!is_array($queued_alerts) || empty($queued_alerts)) {
        return;
    }

    // Determine a common site name/URL for the batch email subject and header.
    // Assumes all alerts in a batch are from the same site (true for a single site mu-plugin).
    $first_alert = reset($queued_alerts);
    $site_name = $first_alert['site_name'] ?? 'Unknown Site';
    $site_url = $first_alert['site_url'] ?? 'Unknown URL';

    // Compile messages into a single body for the email
    $email_body = "The following changes have occurred on your WordPress site '{$site_name}' ({$site_url}):\n\n";
    $email_body .= str_repeat('=', 80) . "\n\n"; // Stronger separator for the whole batch

    foreach ($queued_alerts as $alert) {
        $email_body .= "Event Time: {$alert['timestamp']}\n";
        $email_body .= $alert['message'] . "\n\n";
        $email_body .= str_repeat('-', 80) . "\n\n"; // Separator for each individual event
    }

    $subject = "🔔 Batch WordPress Changes on {$site_name} ({$site_url})";
    $headers = array('X-Enliven-Security-Notice: External Email');

    // Send the compiled email
    $mail_sent = wp_mail(SCM_ALERT_RECIPIENTS, $subject, $email_body, $headers);

    if ($mail_sent) {
        // Clear the transient after successful sending
        delete_transient(SCM_BATCH_TRANSIENT_KEY);
        scm_log_to_file("Successfully sent a batch of " . count($queued_alerts) . " alerts for {$site_name}.");
    } else {
        // Log if sending failed. The transient will expire naturally, or next cron run might retry.
        scm_log_to_file("ERROR: Failed to send batch email alerts for {$site_name}. " . count($queued_alerts) . " alerts remain in queue.");
    }
}


// --- ALL YOUR EXISTING ADD_ACTION CALLS ARE MODIFIED TO USE scm_queue_alert ---

// 🧑 User Login
add_action('wp_login', function($user_login) {
    scm_queue_alert("🧑 User Login: $user_login\nTime: " . current_time('mysql'));
});

// ✅ Plugin Activated by User
add_action('activated_plugin', function($plugin) {
    $username = scm_get_current_username_safe();
    $plugin_data = get_plugin_data(WP_PLUGIN_DIR . '/' . $plugin);
    $plugin_name = isset($plugin_data['Name']) ? $plugin_data['Name'] : $plugin;
    scm_queue_alert("✅ Plugin Activated: \"{$plugin_name}\" by [$username]");
});

// ⚠️ Plugin Deactivated by User
add_action('deactivated_plugin', function($plugin) {
    $username = scm_get_current_username_safe();
    $plugin_data = get_plugin_data(WP_PLUGIN_DIR . '/' . $plugin);
    $plugin_name = isset($plugin_data['Name']) ? $plugin_data['Name'] : $plugin;
    scm_queue_alert("⚠️ Plugin Deactivated: \"{$plugin_name}\" by [$username]");
});

// 🔄 Plugin/Theme Updated
add_action('upgrader_process_complete', function($upgrader, $data) {
    $username = scm_get_current_username_safe();
    if (isset($data['type'])) {
        if ($data['type'] === 'plugin') {
            $updated_plugins = [];
            if (isset($data['plugins']) && is_array($data['plugins'])) {
                foreach ($data['plugins'] as $plugin_path) {
                    $plugin_data = get_plugin_data(WP_PLUGIN_DIR . '/' . $plugin_path);
                    $updated_plugins[] = isset($plugin_data['Name']) ? $plugin_data['Name'] : $plugin_path;
                }
            }
            $plugin_names = !empty($updated_plugins) ? implode(", ", $updated_plugins) : 'Unknown Plugin(s)';
            scm_queue_alert("🔄 Plugin(s) Updated by [$username]: " . $plugin_names);
        }
        if ($data['type'] === 'theme') {
            $updated_themes = [];
            if (isset($data['themes']) && is_array($data['themes'])) {
                foreach ($data['themes'] as $theme_slug) {
                    $theme = wp_get_theme($theme_slug);
                    $updated_themes[] = $theme->get('Name');
                }
            }
            $theme_names = !empty($updated_themes) ? implode(", ", $updated_themes) : 'Unknown Theme(s)';
            scm_queue_alert("🎨 Theme(s) Updated by [$username]: " . $theme_names);
        }
        if ($data['type'] === 'core') {
            scm_queue_alert("⬆️ WordPress Core Updated to version " . get_bloginfo('version') . " by [$username]");
        }
    }
}, 10, 2);

// 🎨 Theme Activated
add_action('switch_theme', function($new_name) {
    $username = scm_get_current_username_safe();
    scm_queue_alert("🎨 Theme Activated: $new_name by [$username]");
});

// 📝 Page/Post/Attachment Created
add_action('save_post', function($post_id, $post, $update) {
    if ($post_id === 0 || wp_is_post_revision($post_id) || wp_is_post_autosave($post_id)) {
        return;
    }
    if ($update) {
        return;
    }
    if (in_array($post->post_type, ['post', 'page', 'attachment'])) {
        $username = scm_get_current_username_safe();
        $post_title = !empty($post->post_title) ? '"' . $post->post_title . '"' : '(No Title)';
        scm_queue_alert("📝 {$post->post_type} Created by [$username]: {$post_title} (ID: $post_id)");
    }
}, 10, 3);

// 📝 Page/Post/Attachment Updated with Details
add_action('post_updated', function($post_id, $post_after, $post_before) {
    if (wp_is_post_revision($post_id) || wp_is_post_autosave($post_id)) {
        return;
    }
    if (!in_array($post_after->post_type, ['post', 'page', 'attachment'])) {
        return;
    }

    $username = scm_get_current_username_safe();
    $post_title = !empty($post_after->post_title) ? '"' . $post_after->post_title . '"' : '(No Title)';
    $message_details = "📝 {$post_after->post_type} Updated by [$username]: {$post_title} (ID: $post_id)";
    $changes = [];

    if ($post_before->post_title !== $post_after->post_title) { $changes[] = "Title changed from '{$post_before->post_title}' to '{$post_after->post_title}'"; }
    if ($post_before->post_content !== $post_after->post_content) { $changes[] = "Content updated"; }
    if ($post_before->post_status !== $post_after->post_status) { $changes[] = "Status changed from '{$post_before->post_status}' to '{$post_after->post_status}'"; }
    if ($post_before->post_name !== $post_after->post_name) { $changes[] = "Slug changed from '{$post_before->post_name}' to '{$post_after->post_name}'"; }
    if ($post_after->post_parent !== $post_before->post_parent) { $old_parent = $post_before->post_parent ? get_the_title($post_before->post_parent) : 'None'; $new_parent = $post_after->post_parent ? get_the_title($post_after->post_parent) : 'None'; $changes[] = "Parent changed from '{$old_parent}' to '{$new_parent}'"; }
    if ($post_before->post_excerpt !== $post_after->post_excerpt) { $changes[] = "Excerpt updated"; }

    if (empty($changes)) { $message_details .= " (Minor update or meta change)"; } else { $message_details .= "\nDetails:\n- " . implode("\n- ", $changes); }
    scm_queue_alert($message_details);
}, 10, 3);


// 🗑️ Page/Post/Custom Post Type Deleted
add_action('before_delete_post', function($post_id) {
    $post = get_post($post_id);
    $username = scm_get_current_username_safe();
    $message_parts = [ 'action' => '🗑️ Deleted:', 'type' => 'Unknown Type', 'title' => 'No Title', 'id' => $post_id, 'user' => $username, ];
    if ($post instanceof WP_Post) { $message_parts['type'] = !empty($post->post_type) ? $post->post_type : 'Unknown Post Type'; $message_parts['title'] = !empty($post->post_title) ? '"' . $post->post_title . '"' : 'No Title'; } else { $message_parts['type'] = 'Invalid Post Object/Non-Standard Deletion'; $message_parts['title'] = 'Cannot retrieve details'; }
    scm_queue_alert("{$message_parts['action']} {$message_parts['type']} - {$message_parts['title']} (ID: {$message_parts['id']}) by [{$message_parts['user']}]");
});


// ❌ Plugin Deletion Detection
add_action('admin_init', function() {
    if (!current_user_can('activate_plugins')) return;
    $previous = get_transient('scm_plugin_list');
    $current = array_keys(get_plugins());
    if ($previous) {
        $deleted = array_diff($previous, $current);
        if (!empty($deleted)) {
            $username = scm_get_current_username_safe();
            $deleted_plugin_names = [];
            foreach ($deleted as $plugin_path) { if (file_exists(WP_PLUGIN_DIR . '/' . $plugin_path)) { $plugin_data = get_plugin_data(WP_PLUGIN_DIR . '/' . $plugin_path); $deleted_plugin_names[] = isset($plugin_data['Name']) ? $plugin_data['Name'] : $plugin_path; } else { $deleted_plugin_names[] = $plugin_path . " (details unavailable)"; } }
            scm_queue_alert("❌ Plugin(s) Deleted by [$username]: " . implode(", ", $deleted_plugin_names));
        }
    }
    set_transient('scm_plugin_list', $current, 12 * HOUR_IN_SECONDS);
});

// ❌ Theme Deletion Detection
add_action('admin_init', function() {
    if (!current_user_can('switch_themes')) return;
    $previous = get_transient('scm_theme_list');
    $current = array_keys(wp_get_themes());
    if ($previous) {
        $deleted = array_diff($previous, $current);
        if (!empty($deleted)) {
            $username = scm_get_current_username_safe();
            $deleted_theme_names = [];
            foreach ($deleted as $theme_slug) { $theme = wp_get_theme($theme_slug); $deleted_theme_names[] = $theme->exists() ? $theme->get('Name') : $theme_slug . " (details unavailable)"; }
            scm_queue_alert("❌ Theme(s) Deleted by [$username]: " . implode(", ", $deleted_theme_names));
        }
    }
    set_transient('scm_theme_list', $current, 12 * HOUR_IN_SECONDS);
});

// 👤 Track Who Created New Users
add_action('user_register', function($user_id) {
    $new_user = get_userdata($user_id);
    $created_user_login = $new_user ? $new_user->user_login : 'Unknown New User';
    $creator_name = scm_get_current_username_safe();
    scm_queue_alert("👤 New User Created: [$created_user_login] (ID: $user_id) by [$creator_name]");
});

// 🗑️ User Deleted
add_action('delete_user', function($user_id) {
    $deleted_user_data = get_userdata($user_id);
    $deleted_username = $deleted_user_data ? $deleted_user_data->user_login : 'Unknown User';
    $deleter_username = scm_get_current_username_safe();
    scm_queue_alert("🗑️ User Deleted: [$deleted_username] (ID: $user_id) by [$deleter_username]");
}, 10, 1);

// 👥 User Profile Updated (Email, Password, Role, etc.)
add_action('profile_update', function($user_id, $old_user_data) {
    $current_user_data = get_userdata($user_id);
    $current_user_login = $current_user_data ? $current_user_data->user_login : 'Unknown User';
    $updater_name = scm_get_current_username_safe();
    $changes = [];
    if ($old_user_data->user_email !== $current_user_data->user_email) { $changes[] = "Email changed from '{$old_user_data->user_email}' to '{$current_user_data->user_email}'"; }
    if (!empty($_POST['pass1'])) { $changes[] = "Password was updated"; }
    if ($old_user_data->display_name !== $current_user_data->display_name) { $changes[] = "Display Name changed from '{$old_user_data->display_name}' to '{$current_user_data->display_name}'"; }
    $old_roles = implode(', ', $old_user_data->roles);
    $new_roles = implode(', ', $current_user_data->roles);
    if ($old_roles !== $new_roles) { $changes[] = "Roles changed from '{$old_roles}' to '{$new_roles}'"; }
    $message = "👥 User Profile Updated: [$current_user_login] (ID: $user_id) by [$updater_name]";
    if (!empty($changes)) { $message .= "\nDetails:\n- " . implode("\n- ", $changes); } else { $message .= " (Minor profile update)"; }
    scm_queue_alert($message);
}, 10, 2);

// 💬 Comment Status Changes
add_action('wp_set_comment_status', function($comment_id, $comment_status) {
    $comment = get_comment($comment_id);
    if (!$comment) return;
    $username = scm_get_current_username_safe();
    $post = get_post($comment->comment_post_ID);
    $post_title = $post ? $post->post_title : 'Unknown Post';
    $comment_author = !empty($comment->comment_author) ? $comment->comment_author : 'Guest';
    $comment_excerpt = wp_trim_words($comment->comment_content, 10, '...');
    scm_queue_alert("💬 Comment Status Changed: \"{$comment_excerpt}\" on \"{$post_title}\" ({$comment->comment_post_ID}) by {$comment_author} to '{$comment_status}' by [$username]");
}, 10, 2);

// 🗑️ Comment Deleted
add_action('delete_comment', function($comment_id) {
    $comment = get_comment($comment_id);
    $username = scm_get_current_username_safe();
    $comment_details = "ID: {$comment_id}";
    if ($comment) { $post = get_post($comment->comment_post_ID); $post_title = $post ? $post->post_title : 'Unknown Post'; $comment_author = !empty($comment->comment_author) ? $comment->comment_author : 'Guest'; $comment_excerpt = wp_trim_words($comment->comment_content, 10, '...'); $comment_details = "on \"{$post_title}\" by {$comment_author}: \"{$comment_excerpt}\" (ID: {$comment_id})"; }
    scm_queue_alert("🗑️ Comment Deleted: {$comment_details} by [$username]");
}, 10, 1);


// 🏷️ Taxonomy Term Created, Edited, Deleted
add_action('create_term', function($term_id, $tt_id, $taxonomy) {
    $term = get_term($term_id, $taxonomy);
    $username = scm_get_current_username_safe();
    scm_queue_alert("🏷️ Term Created: \"{$term->name}\" ({$taxonomy}) by [$username]");
}, 10, 3);

add_action('edit_term', function($term_id, $tt_id, $taxonomy) {
    $term = get_term($term_id, $taxonomy);
    $username = scm_get_current_username_safe();
    scm_queue_alert("🏷️ Term Edited: \"{$term->name}\" ({$taxonomy}) (ID: {$term_id}) by [$username]");
}, 10, 3);

add_action('delete_term', function($term_id, $tt_id, $taxonomy, $deleted_term) {
    $username = scm_get_current_username_safe();
    $term_name = isset($deleted_term->name) ? $deleted_term->name : 'Unknown Term';
    scm_queue_alert("🗑️ Term Deleted: \"{$term_name}\" ({$taxonomy}) (ID: {$term_id}) by [$username]");
}, 10, 4);

// ⚙️ Site Options/Settings Changes
add_action('update_option', function($option_name, $old_value, $new_value) {
    if (!function_exists('wp_get_current_user') || !function_exists('get_bloginfo')) {
        scm_log_to_file("Early option update: {$option_name} changed from '{$old_value}' to '{$new_value}'. Full context not available yet.");
        return;
    }
    $username = scm_get_current_username_safe();
    $tracked_options = [
        'blogname' => 'Site Title', 'blogdescription' => 'Tagline', 'siteurl' => 'WordPress Address (URL)', 'home' => 'Site Address (URL)',
        'admin_email' => 'Administration Email Address', 'date_format' => 'Date Format', 'time_format' => 'Time Format', 'timezone_string' => 'Timezone',
        'default_category' => 'Default Post Category', 'default_post_format' => 'Default Post Format', 'posts_per_page' => 'Blog pages show at most',
        'comments_per_page' => 'Comments per page', 'page_on_front' => 'Homepage displays', 'page_for_posts' => 'Posts page',
        'show_on_front' => 'Reading settings front page display', 'blog_public' => 'Search Engine Visibility',
        'wplacahe_auto_clear_on_update' => 'WP Super Cache: Clear on Update', // Specific to your context, if this is the option
    ];
    if (isset($tracked_options[$option_name])) {
        $setting_name = $tracked_options[$option_name];
        $old_display_value = is_scalar($old_value) ? $old_value : json_encode($old_value);
        $new_display_value = is_scalar($new_value) ? $new_value : json_encode($new_value);
        scm_queue_alert("⚙️ Site Setting Changed: '{$setting_name}' ({$option_name}) from '{$old_display_value}' to '{$new_display_value}' by [$username]");
    }
}, 10, 3);

// 🧩 Widget Changes (Simplified tracking)
add_action('widget_update_callback', function($instance, $new_instance, $old_instance) {
    $username = scm_get_current_username_safe();
    scm_queue_alert("🧩 Widget Settings Updated by [$username]");
    return $instance;
}, 10, 3);

add_action('sidebar_admin_setup', function() {
    $username = scm_get_current_username_safe();
    if (isset($_POST['sidebar_widgets'])) {
        $old_widgets = get_option('sidebars_widgets');
        $new_widgets = $_POST['sidebar_widgets'];
        if (json_encode($old_widgets) !== json_encode($new_widgets)) {
             scm_queue_alert("🧩 Widgets Reordered, Added, or Deleted in Sidebars by [$username]");
        }
    }
});


// 📦 Monthly Log Rotation
add_action('init', function() {
    if (!function_exists('file_exists') || !defined('WP_CONTENT_DIR') || !defined('DAY_IN_SECONDS')) {
        return;
    }
    $logfile = WP_CONTENT_DIR . '/scm-change-log.log';
    if (file_exists($logfile)) {
        $ageInSeconds = time() - filemtime($logfile);
        if ($ageInSeconds > 30 * DAY_IN_SECONDS) {
            $archive = WP_CONTENT_DIR . '/scm-change-log-' . date('Y-m-d') . '.log';
            if (rename($logfile, $archive)) {
                scm_queue_alert("📦 Log rotated: $archive");
            } else {
                scm_log_to_file("Failed to rotate log file: $logfile");
            }
        }
    }
});

// 🔐 Create Limited Admin Role
add_action('init', function() {
    if (!function_exists('get_role')) {
        return;
    }
    if (!get_role('limited_admin')) {
        $admin = get_role('administrator');
        if ($admin) {
            add_role('limited_admin', 'Limited Admin', $admin->capabilities);
            $limited_admin = get_role('limited_admin');
            if ($limited_admin) {
                $limited_admin->remove_cap('edit_users');
                $limited_admin->remove_cap('create_users');
                $limited_admin->remove_cap('delete_users');
                $limited_admin->remove_cap('promote_users');
            }
        }
    }
});